<?php
require_once __DIR__ . '/../config.php';
require_once __DIR__ . '/_token.php';
require_once __DIR__ . '/_ls.php';
require_once __DIR__ . '/_entitlements.php';

// CORS + preflight
if (($_SERVER['REQUEST_METHOD'] ?? '') === 'OPTIONS') {
  header('Access-Control-Allow-Origin: *');
  header('Access-Control-Allow-Methods: POST, OPTIONS');
  header('Access-Control-Allow-Headers: Content-Type');
  http_response_code(204);
  exit;
}

if (($_SERVER['REQUEST_METHOD'] ?? '') !== 'POST') {
  json_response(['ok' => false, 'error' => 'METHOD_NOT_ALLOWED'], 405);
}

header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

$body = json_decode(file_get_contents('php://input'), true);
$token = (string)($body['token'] ?? '');
$instance_id = (string)($body['instance_id'] ?? '');

$email = auth_email_from_token($token);
if (!$email) {
  json_response(['ok' => false, 'error' => 'UNAUTHORIZED'], 401);
}

// 1) Whitelist overrides everything
// whitelist.json is expected to be: {"email@x.com": {"label":"...","features":["fun1"],"expires_at":null|unix}}
$wlPath = DATA_DIR . '/whitelist.json';
$wl = read_json($wlPath, []);
$wlRec = $wl[$email] ?? null;

$wlActive = false;
$wlFeatures = [];

if (is_array($wlRec)) {
  $exp = $wlRec['expires_at'] ?? null;
  $wlActive = ($exp === null) ? true : (now() <= (int)$exp);
  $wlFeatures = $wlRec['features'] ?? ($wlRec['allowed'] ?? []);
  if (!is_array($wlFeatures)) $wlFeatures = [];
}

if ($wlActive) {
  json_response([
    'ok' => true,
    'entitlements' => [
      'source' => 'whitelist',
      'email' => $email,
      'plan' => 'WHITELIST',
      'allowed' => array_values(array_unique(array_map('strval', $wlFeatures))),
      'paid' => ['active' => false],
      'trial' => ['exists' => false, 'active' => false, 'seconds_left' => 0]
    ]
  ]);
}

// 2) Paid plan (if we have a stored license)
$paidPath = DATA_DIR . '/paid.json';
$paid = read_json($paidPath, []);
$rec = $paid[$email] ?? null;
if (is_array($rec) && !empty($rec['license_key'])) {
  $lk = (string)$rec['license_key'];
  $iid = $instance_id ?: (string)($rec['instance_id'] ?? '');

  $v = ls_validate($lk, $iid);
  if ($v['ok'] && isset($v['data']['valid'])) {
    $valid = !!$v['data']['valid'];
    $meta = $v['data']['meta'] ?? [];
    $variant_id = (int)($meta['variant_id'] ?? ($rec['variant_id'] ?? 0));
    $plan = plan_from_variant_id($variant_id);

    // basic security: email match
    $lsEmail = normalize_email($meta['customer_email'] ?? '');
    if ($lsEmail && $lsEmail !== $email) {
      // don't delete paid record, but refuse
      json_response(['ok' => true, 'entitlements' => [
        'source' => 'none',
        'email' => $email,
        'plan' => null,
        'allowed' => [],
        'paid' => ['active' => false, 'error' => 'LICENSE_EMAIL_MISMATCH'],
        'trial' => ['exists' => true, 'active' => false, 'seconds_left' => 0]
      ]]);
    }

    if ($valid && $plan) {
      $allowed = allowed_from_plan($plan);
      // update cached fields
      $paid[$email]['variant_id'] = $variant_id;
      $paid[$email]['plan'] = $plan;
      if (!empty($v['data']['instance']['id'])) $paid[$email]['instance_id'] = (string)$v['data']['instance']['id'];
      $paid[$email]['last_check'] = now();
      write_json($paidPath, $paid);

      json_response([
        'ok' => true,
        'entitlements' => [
          'source' => 'paid',
          'email' => $email,
          'plan' => $plan,
          'allowed' => $allowed,
          'paid' => [
            'active' => true,
            'variant_id' => $variant_id,
            'expires_at' => $v['data']['license_key']['expires_at'] ?? null,
          ],
          'trial' => ['exists' => true, 'active' => false, 'seconds_left' => 0]
        ]
      ]);
    }
  }

  // paid exists but invalid / unknown plan
  // fall through to trial (or lock)
}

// 3) Trial status (existing endpoints already handle TTL + 15min)
// We reuse your trial/status.php to keep logic in one place.
$trialEndpoint = __DIR__ . '/../trial/status.php';
if (file_exists($trialEndpoint)) {
  // We can't include it safely (it may exit). Instead: implement tiny trial check here.
}

// Minimal trial check (expects data/trials.json format: {email:{start,expires}})
$trialPath = DATA_DIR . '/trials.json';
$trials = read_json($trialPath, []);
$t = $trials[$email] ?? null;
$exists = is_array($t);
$active = false;
$left = 0;
if ($exists) {
  $exp = (int)($t['expires_at'] ?? $t['expires'] ?? 0);
  if ($exp > 0) {
    $left = max(0, $exp - now());
    $active = $left > 0;
  }
}

json_response([
  'ok' => true,
  'entitlements' => [
    'source' => $active ? 'trial' : 'none',
    'email' => $email,
    'plan' => $active ? 'TRIAL' : null,
    'allowed' => $active ? ['fun1','fun2','fun3'] : [],
    'paid' => ['active' => false],
    'trial' => ['exists' => $exists, 'active' => $active, 'seconds_left' => $left]
  ]
]);
